/** @file   Gradient.h
 * @brief   Implementation of Gradient class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:38 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_GRADIENT_H
#define H_GRADIENT_H

#include <string>
#include "eng2d_dll.h"
#include "Color.h"


namespace eng2d {


/** @class  Gradient
 * @brief   A color gradient with 3 colors and biasing point.
 * @author  Tomi Lamminsaari
 *
 * Gradient - class represents a color gradients. These gradients consists
 * of 3 colors and a biasing point. We call these three colors "left color",
 * "middle color" and "right color". Biasingpoint defines the position of
 * the middle color between the "left" and "right" colors.
 * 
 * Biasingpoint is defined by percents where value 50 means that the
 * transition from "left color" to "middle color" is as long as the
 * transition from "middle color" to "right color". If biasing is less
 * than 50, the first transition is sharper than the another.
 */
class DLLIMPORT Gradient
{
public:

  ///
  /// Static members and methods
  /// ==========================
  

  
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Default constructor. Creates smooth gradient from black to white.
   */
  Gradient();
  
  /** Constructs a gradient from 'c1' to 'c2'.
   * @param     c1                The left color.
   * @param     c2                The right color.
   */
  Gradient(Color c1, Color c2);
  
  /** Constructs the gradient from 'c1' to 'c3' via 'c2'.
   * @param     c1                The left color.
   * @param     c2                The middle color.
   * @param     c3                The right color.
   */
  Gradient(Color c1, Color c2, Color c3);
  
  /** Constructs the gradient from 'c1' to 'c3' via 'c2' with bias-point.
   * @param     c1                The left color.
   * @param     c2                The middle color.
   * @param     c3                The right color.
   * @param     bias              The biasing point in percents ( 1..99 )
   */
  Gradient(Color c1, Color c2, Color c3, int bias);
  
  /** A copy constructor.
   */
  Gradient(const Gradient& rO);
  
  /** Destructor.
   */
  ~Gradient();
  
  /** Assignment operator
   */
  Gradient& operator = (const Gradient& rO);


  ///
  /// Methods
  /// =======
  
  /** Sets the left color.
   * @param     c                 The left color.
   */
  void setC1(Color c);
  
  /** Sets the middle color.
   * @param     c                 The middle color.
   */
  void setC2(Color c);
  
  /** Sets the right color.
   * @param     c                 The right color.
   */
  void setC3(Color c);
  
  /** Sets the biasing point.
   * @param     bias              THe biasingpoint in percents ( 1..99 )
   */
  void setBias(float bias);
  
    
  ///
  /// Getter methods
  /// ==============
  
  /** Applies the gradient. Returns the color on position 'pos' if the
   * whole gradient is stretched from 'rangeStart' to 'rangeEnd'.
   * <p>
   * Remember the rule: rangeStart <= pos <= rangeEnd
   * @param     pos               The position where the color is being
   *                              calculated.
   * @param     rangeStart        The start value. Should be less or equal
   *                              to 'pos'.
   * @param     rangeEnd          The end value. Should be bigger or equal
   *                              to 'pos'.
   * @return    The color on requested position from the gradient.
   */
  
  Color getColorAt(int pos, int rangeStart, int rangeEnd) const;
  
  /** Returns the left color.
   * @return    The left color.
   */
  Color getC1() const;
  
  /** Returns the middle color.
   * @return    The middle color.
   */
  Color getC2() const;
  
  /** Returns the right color.
   * @return    The right color.
   */
  Color getC3() const;
  
  /** Returns the biasing point.
   * @return    The biasing point.
   */
  float   getBias() const;
  
  /** Calculates a color at @c pos if this gradient is stretched within the
   * @c lowValue and @c highValue. So the remember this:
   * lowValue <= pos <= highValue.
   * @param[in] pos               From which position we should take the
   *                              color from the gradient. If less than
   *                              @c lowValue - parameter we use the
   *                              lowValue. If greater than highValue - parameter
   *                              we use highValue.
   * @param[in] lowValue          Defines the lower bound of the gradient
   *                              range. Should be less or equal to
   *                              @c pos - parameter.
   * @param[in] highValue         Defines the higher bound of the gradient
   *                              range. Should be greater or equal to
   *                              @c pos - parameter.
   * @param[out] pC               Pointer to an instance of Color where we
   *                              store the calculated color.
   */
  void color( float pos, float lowValue, float highValue, Color* pC ) const;
  
protected:

  
  ///
  /// Members
  /// =======
  
  /** The left color. */
  Color   m_begCol;
  /** The middle color. */
  Color   m_midCol;
  /** The right color. */
  Color   m_endCol;
  /** The biasingpoint. */
  float     m_bias;
};

};  // end of namespace


#endif
